--[[
Copyright (C) GtX (Andy), 2019

Author: GtX | Andy
Date: 04.01.2019
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]


TransferOwnershipPoint = {}
TransferOwnershipPoint.MOD_NAME = g_currentModName

local TransferOwnershipPoint_mt = Class(TransferOwnershipPoint)

function TransferOwnershipPoint.registerXMLPaths(schema, basePath)
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".transferOwnership#playerTriggerNode", "Player trigger node")
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".transferOwnership#objectsTriggerNode", "Objects trigger node")
end

function TransferOwnershipPoint.new()
    local self = setmetatable({}, TransferOwnershipPoint_mt)

    self.objectShapesInRange = {}
    self.isEnabled = true

    return self
end

function TransferOwnershipPoint:load(components, xmlFile, key, i3dMappings)
    self.playerTrigger = xmlFile:getValue(key .. ".transferOwnership#playerTriggerNode", nil, components, i3dMappings)

    if not CollisionFlag.getHasMaskFlagSet(self.playerTrigger, CollisionFlag.PLAYER) then
        Logging.xmlWarning(xmlFile, "Missing collision mask bit '%d'. Please add this bit to objects transfer player trigger node '%s'", CollisionFlag.getBit(CollisionFlag.PLAYER), I3DUtil.getNodePath(self.playerTrigger))
    end

    self.objectsTriggerNode = xmlFile:getValue(key .. ".transferOwnership#objectsTriggerNode", nil, components, i3dMappings)

    if not CollisionFlag.getHasMaskFlagSet(self.objectsTriggerNode, CollisionFlag.VEHICLE) then
        Logging.xmlWarning(xmlFile, "Missing collision mask bit '%d'. Please add this bit to objects transfer area trigger node '%s'", CollisionFlag.getBit(CollisionFlag.VEHICLE), I3DUtil.getNodePath(self.objectsTriggerNode))
    elseif not CollisionFlag.getHasMaskFlagSet(self.objectsTriggerNode, CollisionFlag.DYNAMIC_OBJECT) then
        Logging.xmlWarning(xmlFile, "Missing collision mask bit '%d'. Please add this bit to objects transfer area trigger node '%s'", CollisionFlag.getBit(CollisionFlag.DYNAMIC_OBJECT), I3DUtil.getNodePath(self.objectsTriggerNode))
    end

    self.activatable = TransferOwnershipPointActivatable.new(self)

    return true
end

function TransferOwnershipPoint:finalizePlacement()
    if self.playerTrigger ~= nil then
        addTrigger(self.playerTrigger, "playerTriggerCallback", self)
    end

    if self.objectsTriggerNode ~= nil then
        addTrigger(self.objectsTriggerNode, "transferAreaTriggerCallback", self)
    end
end

function TransferOwnershipPoint:delete()
    self.isEnabled = false

    if self.playerTrigger ~= nil then
        removeTrigger(self.playerTrigger)

        self.playerTrigger = nil
    end

    if self.objectsTriggerNode ~= nil then
        removeTrigger(self.objectsTriggerNode)

        self.objectShapesInRange = {}
        self.objectsTriggerNode = nil

        if g_transferOwnershipScreen ~= nil then
            g_transferOwnershipScreen:removeTransferPoint(self)
        end
    end

    g_currentMission.activatableObjectsSystem:removeActivatable(self.activatable)
    self.activatable = nil
end

function TransferOwnershipPoint:openMenu()
    if g_currentMission:getHasPlayerPermission("transferMoney") then
        TransferOwnershipScreen.show(self)
    else
        local text = string.format("%s\n\n%s", g_i18n:getText("shop_messageNoPermissionGeneral"), g_i18n:getText("ui_farmManager"))
        InfoDialog.show(text, nil, nil, DialogElement.TYPE_INFO, g_i18n:getText("button_back"), InputAction.MENU_BACK)
    end
end

function TransferOwnershipPoint:determineCurrentObjects()
    local vehicles = {}
    local bales = {}

    for shapeId, inRange in pairs(self.objectShapesInRange) do
        if inRange ~= nil and entityExists(shapeId) then
            local object = g_currentMission.nodeToObject[shapeId]

            if object ~= nil and object.isa ~= nil and object.ownerFarmId ~= nil and object.ownerFarmId ~= FarmManager.SPECTATOR_FARM_ID then
                if object:isa(Vehicle) then
                    if not object.isDeleted and not SpecializationUtil.hasSpecialization(Rideable, object.specializations) and object.propertyState == VehiclePropertyState.OWNED and self:getNodeInTriggerRange(object.rootNode) then
                        table.addElement(vehicles, object)
                    else
                        self.objectShapesInRange[shapeId] = nil
                    end
                elseif object:isa(Bale) and object:getCanBeSold() and object.dynamicMountJointIndex == nil and self:getNodeInTriggerRange(object.nodeId) then
                    table.addElement(bales, object)
                else
                    self.objectShapesInRange[shapeId] = nil
                end
            else
                self.objectShapesInRange[shapeId] = nil
            end
        else
            self.objectShapesInRange[shapeId] = nil
        end
    end

    return vehicles, bales
end

function TransferOwnershipPoint:getNodeInTriggerRange(node)
    if node ~= nil and self.objectsTriggerNode ~= nil then
        -- Try and avoid teleport issues, should never teleport from a trigger but some people do not know this.
        return calcDistanceFrom(node, self.objectsTriggerNode) < 25
    end

    return false
end

function TransferOwnershipPoint:playerTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
    if self.isEnabled and (onEnter or onLeave) and g_localPlayer ~= nil and otherId == g_localPlayer.rootNode then
        if onEnter then
            g_currentMission.activatableObjectsSystem:addActivatable(self.activatable)
        else
            g_currentMission.activatableObjectsSystem:removeActivatable(self.activatable)
        end
    end
end

function TransferOwnershipPoint:transferAreaTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
    if otherShapeId ~= nil and (onEnter or onLeave) then
        if onEnter then
            self.objectShapesInRange[otherShapeId] = true
        elseif onLeave then
            self.objectShapesInRange[otherShapeId] = nil
        end

        if g_transferOwnershipScreen ~= nil then
            g_transferOwnershipScreen:updateObjects(self)
        end
    end
end

TransferOwnershipPointActivatable = {}
local TransferOwnershipPointActivatable_mt = Class(TransferOwnershipPointActivatable)

function TransferOwnershipPointActivatable.new(transferPoint)
    local self = setmetatable({}, TransferOwnershipPointActivatable_mt)

    self.transferPoint = transferPoint
    self.activateText = g_i18n:getText("input_MENU")

    return self
end

function TransferOwnershipPointActivatable:getIsActivatable()
    if not self.transferPoint.isEnabled then
        return false
    end

    if g_localPlayer == nil or g_localPlayer:getIsInVehicle() then
        return false
    end

    return g_currentMission:getFarmId() ~= FarmManager.SPECTATOR_FARM_ID
end

function TransferOwnershipPointActivatable:run()
    self.transferPoint:openMenu()
end

function TransferOwnershipPointActivatable:getDistance(x, y, z)
    local tx, _, tz = getWorldTranslation(self.transferPoint.playerTrigger)

    return MathUtil.getPointPointDistance(tx, tz, x, z)
end
